<?php

use App\Http\Controllers\Authentication\CheckController;
use App\Http\Controllers\Authentication\LoginController;
use App\Http\Controllers\Authentication\RefreshController;
use App\Http\Controllers\Base\ContentController;
use App\Http\Controllers\Base\DefaultController;
use App\Http\Controllers\Agent\StatusController;
use App\Http\Controllers\License\ValidateController;
use App\Http\Controllers\Inquiry\GetFormOptionsController;
use App\Http\Controllers\Inquiry\GetSupportedPlansController;
use App\Http\Controllers\Inquiry\SubmitInquityController;
use App\Http\Controllers\Inquiry\CalculateOutboundPremiumController;
use App\Http\Controllers\Inquiry\SubmitOutboundInquiryController;
use App\Http\Controllers\Inquiry\GetQuestionnaireController;
use App\Http\Controllers\Inquiry\SubmitQuestionnaireController;
use Illuminate\Support\Facades\Route;

Route::middleware(['api.header'])->group(function () {
    // Authentication
    Route::prefix('auth')->group(function () {
        Route::get('check', CheckController::class)->middleware(['auth.check']);
        Route::post('login', LoginController::class);
        Route::patch('refresh', RefreshController::class)->middleware(['auth.check']);
    });

    // Base endpoints
    Route::prefix('base')->group(function () {
        Route::get('health', DefaultController::class);
        Route::get('content', ContentController::class)->middleware(['auth.check']);
    });

    // License management
    Route::prefix('licenses')->group(function () {
        Route::post('validate', ValidateController::class);
    });

    // Agent management
    Route::prefix('agents')->group(function () {
        Route::get('status', StatusController::class);
    });

    // Insurance inquiry endpoints
    Route::prefix('inquiry')->group(function () {
        Route::get('/', GetSupportedPlansController::class)->middleware(['auth.check']);
        Route::post('/', SubmitInquityController::class)->middleware(['auth.check']);
        Route::options('/', GetFormOptionsController::class)->middleware(['auth.check']);

        // Outbound from Canada insurance
        Route::prefix('outbound')->group(function () {
            Route::get('premium', CalculateOutboundPremiumController::class)->middleware(['auth.check']);
            Route::post('/', SubmitOutboundInquiryController::class)->middleware(['auth.check']);
        });

        // Medical Questionnaire
        Route::prefix('questionnaire')->group(function () {
            Route::get('/', GetQuestionnaireController::class)->middleware(['auth.check']);
            Route::post('/', SubmitQuestionnaireController::class)->middleware(['auth.check']);
            Route::post('/bulk', [SubmitQuestionnaireController::class, 'bulkSubmit'])->middleware(['auth.check']);
            Route::get('/status/{inquiryId}', [GetQuestionnaireController::class, 'getStatusOverview'])->middleware(['auth.check']);
        });
    });

    // Dev endpoints (local only)
    if (app()->environment('local', 'development')) {
        Route::prefix('dev')->group(function () {
            Route::get('test-rates', function () {
                return response()->json([
                    'message' => 'Test endpoints for rate calculation',
                    'available_tests' => [
                        'outbound_premium' => 'POST /api/inquiry/outbound/premium',
                        'questionnaire' => 'POST /api/inquiry/questionnaire',
                    ]
                ]);
            });
        });
    }
});

// Public endpoints
Route::middleware(['api.header'])->group(function () {
    Route::get('status', function () {
        return response()->json([
            'status' => 'operational',
            'service' => 'Insurance Platform API',
            'version' => '2.0.0',
            'timestamp' => now()->toISOString(),
            'endpoints' => [
                'authentication' => [
                    'POST /api/auth/login' => 'Login',
                    'GET /api/auth/check' => 'Check authentication (authenticated)',
                    'PATCH /api/auth/refresh' => 'Refresh token (authenticated)',
                ],
                'insurance' => [
                    'GET /api/inquiry' => 'Get supported plans (authenticated)',
                    'POST /api/inquiry' => 'Submit inquiry (authenticated)',
                    'GET /api/inquiry/outbound/premium' => 'Calculate outbound premium (authenticated)',
                    'POST /api/inquiry/outbound' => 'Submit outbound inquiry (authenticated)',
                    'GET /api/inquiry/questionnaire' => 'Get medical questionnaire (authenticated)',
                    'POST /api/inquiry/questionnaire' => 'Submit questionnaire answers (authenticated)',
                ],
                'management' => [
                    'POST /api/licenses/validate' => 'Validate license',
                    'GET /api/agents/status' => 'Get agent status',
                    'GET /api/base/content' => 'Get content (authenticated)',
                ]
            ]
        ]);
    });
});

// 404 handler
Route::fallback(function () {
    return response()->json([
        'error' => 'Route not found',
        'message' => 'The requested API endpoint does not exist.',
        'documentation' => 'Check GET /api/status for available endpoints',
        'timestamp' => now()->toISOString(),
    ], 404);
});
