# Docker Setup Guide

This guide explains how to run the Insurance application using Docker.

## Prerequisites

- Docker Engine 20.10+
- Docker Compose 2.0+

## Quick Start

### 1. Build and Start Containers

```bash
docker compose up -d --build
```

**Note**: This guide uses `docker compose` (v2 syntax). If you're using Docker Compose v1, replace `docker compose` with `docker-compose`.

This will start:
- **insurance_app**: PHP-FPM application container
- **insurance_db**: MySQL 8.0 database
- **nginx**: Web server

### 2. Install Dependencies

```bash
# Install PHP dependencies
docker compose exec insurance_app composer install

# Install Node dependencies
docker compose exec insurance_app npm install
```

### 3. Configure Environment

```bash
# Copy environment file (if .env.example exists)
docker compose exec insurance_app cp .env.example .env

# Generate application key
docker compose exec insurance_app php artisan key:generate

# Configure database connection in .env:
# DB_CONNECTION=mysql
# DB_HOST=insurance_db
# DB_PORT=3306
# DB_DATABASE=insurance
# DB_USERNAME=insurance
# DB_PASSWORD=insurance
```

### 4. Run Migrations

```bash
docker compose exec insurance_app php artisan migrate
```

### 5. Build Frontend Assets

```bash
# Development
docker compose exec insurance_app npm run dev

# Production
docker compose exec insurance_app npm run build
```

## Common Commands

### Access Application Container

```bash
docker compose exec insurance_app bash
```

### Run Artisan Commands

```bash
docker compose exec insurance_app php artisan <command>
```

Examples:
```bash
# Run migrations
docker compose exec insurance_app php artisan migrate

# Run tests
docker compose exec insurance_app php artisan test

# Clear cache
docker compose exec insurance_app php artisan cache:clear
```

### View Logs

```bash
# All services
docker compose logs -f

# Specific service
docker compose logs -f insurance_app
docker compose logs -f nginx
docker compose logs -f insurance_db
```

### Stop Containers

```bash
docker compose stop
```

### Start Containers

```bash
docker compose start
```

### Restart Containers

```bash
docker compose restart
```

### Remove Containers

```bash
docker compose down
```

### Remove Containers and Volumes

```bash
docker compose down -v
```

## Database Access

### MySQL Client

```bash
docker compose exec insurance_db mysql -u insurance -pinsurance insurance
```

### Connection Details

- **Host**: `insurance_db` (from within containers) or `localhost` (from host)
- **Port**: `3306`
- **Database**: `insurance`
- **Username**: `insurance`
- **Password**: `insurance`
- **Root Password**: `root`

## Network Configuration

The setup uses three networks:

1. **insurance-internal**: Internal communication between services
2. **shared-services**: External network for shared services (if exists)
3. **proxy**: External network for nginx-proxy (if exists)

## Volumes

- **Project files**: Mounted from current directory
- **composer_cache**: Cached Composer packages
- **npm_cache**: Cached npm packages
- **insurance_db_data**: MySQL data persistence

## Access URLs

- **Application**: http://localhost (via nginx-proxy) or http://insurance.local
- **API**: http://localhost/api
- **Admin Panel**: http://localhost/admin

## Troubleshooting

### Permission Issues

```bash
# Fix storage permissions
docker compose exec insurance_app chmod -R 775 storage bootstrap/cache
docker compose exec insurance_app chown -R www-data:www-data storage bootstrap/cache
```

### Rebuild Containers

```bash
docker compose down
docker compose build --no-cache
docker compose up -d
```

### Database Connection Issues

1. Ensure database container is running: `docker compose ps`
2. Check database logs: `docker compose logs insurance_db`
3. Verify .env configuration matches docker-compose.yml

### Clear All Caches

```bash
docker compose exec insurance_app php artisan optimize:clear
```

## Development Workflow

For active development, you can run:

```bash
# Start services
docker compose up -d

# In separate terminals:
# Terminal 1: Watch logs
docker compose logs -f insurance_app

# Terminal 2: Run queue worker
docker compose exec insurance_app php artisan queue:work

# Terminal 3: Watch frontend assets
docker compose exec insurance_app npm run dev
```

## Production Considerations

For production deployment:

1. Update `.env` with production values
2. Set `APP_ENV=production` and `APP_DEBUG=false`
3. Run `php artisan config:cache` and `php artisan route:cache`
4. Build production assets: `npm run build`
5. Review security settings in `php.ini` and `nginx.conf`
6. Use proper SSL/TLS certificates
7. Configure proper resource limits in `docker-compose.yml`

