<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        // 1. اضافه کردن فیلدها به جدول inquiries
        Schema::table('inquiries', function (Blueprint $table) {
            // فیلدهای مربوط به مقصد و مدت سفر
            $table->enum('destination', ['canada', 'worldwide_excluding_usa', 'worldwide_including_usa'])
                  ->nullable()
                  ->after('insurance_cover')
                  ->comment('Destination for travel insurance');
            
            $table->unsignedSmallInteger('trip_duration')
                  ->nullable()
                  ->after('destination')
                  ->comment('Trip duration in days (including departure and return days)');
            
            // فیلدهای محاسبات مالی
            $table->decimal('base_premium', 12, 2)
                  ->nullable()
                  ->after('trip_duration')
                  ->comment('Base premium before discounts and surcharges');
            
            $table->decimal('total_premium', 12, 2)
                  ->nullable()
                  ->after('base_premium')
                  ->comment('Total premium after all calculations');
            
            $table->decimal('tax_amount', 12, 2)
                  ->default(0)
                  ->after('total_premium')
                  ->comment('Tax amount');
            
            $table->decimal('final_amount', 12, 2)
                  ->nullable()
                  ->after('tax_amount')
                  ->comment('Final payable amount');
            
            // فیلدهای مربوط به شرایط خاص
            $table->boolean('canada_half_price_applied')
                  ->default(false)
                  ->after('final_amount')
                  ->comment('Whether Canada half-price rule was applied');
            
            $table->boolean('has_smoking_surcharge')
                  ->default(false)
                  ->after('canada_half_price_applied')
                  ->comment('Whether smoking surcharge was applied');
            
            // فیلدهای مربوط به پرسشنامه
            $table->boolean('questionnaire_required')
                  ->default(false)
                  ->after('has_smoking_surcharge')
                  ->comment('Whether medical questionnaire is required');
            
            $table->boolean('questionnaire_completed')
                  ->default(false)
                  ->after('questionnaire_required')
                  ->comment('Whether questionnaire is completed');
            
            $table->timestamp('questionnaire_completed_at')
                  ->nullable()
                  ->after('questionnaire_completed')
                  ->comment('When questionnaire was completed');
            
            // ایندکس‌ها برای بهینه‌سازی جستجو
            $table->index(['destination', 'is_entry']);
            $table->index(['questionnaire_required', 'questionnaire_completed']);
        });

        // 2. اضافه کردن فیلدها به جدول people
        Schema::table('people', function (Blueprint $table) {
            $table->decimal('individual_premium', 12, 2)
                  ->nullable()
                  ->after('is_male')
                  ->comment('Calculated premium for this individual');
            
            $table->enum('rate_category', ['A', 'B', 'C'])
                  ->nullable()
                  ->after('individual_premium')
                  ->comment('Rate category for age 60+ (Manulife specific)');
            
            $table->boolean('has_smoking_surcharge')
                  ->default(false)
                  ->after('rate_category')
                  ->comment('Whether smoking surcharge applies to this person');
            
            $table->boolean('is_eligible')
                  ->default(true)
                  ->after('has_smoking_surcharge')
                  ->comment('Whether this person is eligible for insurance');
            
            $table->text('ineligibility_reason')
                  ->nullable()
                  ->after('is_eligible')
                  ->comment('Reason for ineligibility if applicable');
            
            $table->index(['rate_category', 'is_eligible']);
        });

        // 3. اضافه کردن فیلدها به جدول plans
        Schema::table('plans', function (Blueprint $table) {
            $table->boolean('requires_questionnaire')
                  ->default(false)
                  ->after('is_active')
                  ->comment('Whether this plan requires medical questionnaire');
            
            $table->string('questionnaire_type', 50)
                  ->nullable()
                  ->after('requires_questionnaire')
                  ->comment('Type of questionnaire (e.g., manulife_60_plus)');
            
            $table->json('rate_tables')
                  ->nullable()
                  ->after('questionnaire_type')
                  ->comment('Rate tables in JSON format for complex calculations');
            
            $table->boolean('supports_outbound')
                  ->default(false)
                  ->after('rate_tables')
                  ->comment('Whether this plan supports outbound Canada insurance');
            
            $table->boolean('supports_inbound')
                  ->default(true)
                  ->after('supports_outbound')
                  ->comment('Whether this plan supports inbound Canada insurance');
            
            $table->index(['requires_questionnaire', 'questionnaire_type']);
            $table->index(['supports_outbound', 'supports_inbound']);
        });
    }

    public function down(): void
    {
        Schema::table('plans', function (Blueprint $table) {
            $table->dropColumn([
                'requires_questionnaire',
                'questionnaire_type',
                'rate_tables',
                'supports_outbound',
                'supports_inbound',
            ]);
            $table->dropIndex(['requires_questionnaire', 'questionnaire_type']);
            $table->dropIndex(['supports_outbound', 'supports_inbound']);
        });

        Schema::table('people', function (Blueprint $table) {
            $table->dropColumn([
                'individual_premium',
                'rate_category',
                'has_smoking_surcharge',
                'is_eligible',
                'ineligibility_reason',
            ]);
            $table->dropIndex(['rate_category', 'is_eligible']);
        });

        Schema::table('inquiries', function (Blueprint $table) {
            $table->dropColumn([
                'destination',
                'trip_duration',
                'base_premium',
                'total_premium',
                'tax_amount',
                'final_amount',
                'canada_half_price_applied',
                'has_smoking_surcharge',
                'questionnaire_required',
                'questionnaire_completed',
                'questionnaire_completed_at',
            ]);
            $table->dropIndex(['destination', 'is_entry']);
            $table->dropIndex(['questionnaire_required', 'questionnaire_completed']);
        });
    }
};