<?php

namespace App\Filament\Admin\Resources\InquiryResource\Pages;

use App\Filament\Admin\Resources\InquiryResource;
use Filament\Actions;
use Filament\Resources\Pages\ListRecords;
use Illuminate\Database\Eloquent\Builder;

/**
 * List Inquiries Page
 * 
 * This page displays all insurance inquiries in a table view.
 * It includes header actions for creating new inquiries and exporting data,
 * as well as bulk actions for managing multiple records at once.
 */
class ListInquiries extends ListRecords
{
    /**
     * The resource class associated with this page
     */
    protected static string $resource = InquiryResource::class;

    /**
     * Header actions displayed at the top of the page
     * 
     * These actions provide quick access to common operations
     * like creating new records or exporting data.
     */
    protected function getHeaderActions(): array
    {
        return [
            // Action 1: Create new inquiry
            Actions\CreateAction::make()
                ->label('New Inquiry')
                ->icon('heroicon-o-plus-circle')
                ->color('primary')
                ->tooltip('Create a new insurance inquiry')
                ->modalHeading('Create New Inquiry')
                ->modalDescription('Fill in the details for a new insurance inquiry')
                ->modalSubmitActionLabel('Create Inquiry')
                ->modalCancelActionLabel('Cancel'),
            
            // Action 2: Export inquiries to Excel
            Actions\Action::make('export')
                ->label('Export to Excel')
                ->icon('heroicon-o-arrow-down-tray')
                ->color('success')
                ->tooltip('Export all inquiries to Excel format')
                ->action(function () {
                    // Export logic will be implemented here
                    // This can use Laravel Excel package or custom export
                    // For now, it's a placeholder that shows a notification
                    $this->notify('success', 'Export feature will be implemented soon.');
                })
                ->requiresConfirmation()
                ->modalHeading('Export Inquiries')
                ->modalDescription('This will export all visible inquiries to an Excel file.')
                ->modalSubmitActionLabel('Export')
                ->modalCancelActionLabel('Cancel'),
            
            // Action 3: Refresh the table data
            Actions\Action::make('refresh')
                ->label('Refresh')
                ->icon('heroicon-o-arrow-path')
                ->color('gray')
                ->tooltip('Refresh the table data')
                ->action(fn () => $this->refreshTableData())
                ->outlined(),
        ];
    }

    /**
     * Get the table query with default filters applied
     * 
     * This method can be used to apply default filters or scopes
     * to the query that fetches records for the table.
     * 
     * @return Builder
     */
    protected function getTableQuery(): Builder
    {
        return parent::getTableQuery()
            // Ensure we load the agent relationship for display
            ->with(['agent'])
            // Apply any default ordering or filters here
            ->orderBy('inquired_at', 'desc');
    }

    /**
     * Get the table records eager loaded with necessary relationships
     * 
     * This optimizes performance by loading relationships upfront
     * instead of making separate queries for each record.
     * 
     * @return Builder
     */
    protected function getTableQueryForExport(): Builder
    {
        return $this->getTableQuery()
            ->with([
                'agent',
                'people', // Include insured persons for export
                'city', // Include city information
            ]);
    }

    /**
     * Refresh the table data and notify the user
     * 
     * This method clears any cached data and refreshes the table
     * while providing user feedback.
     */
    protected function refreshTableData(): void
    {
        $this->resetTable();
        $this->notify('success', 'Table data refreshed successfully.');
    }

    /**
     * Get the empty state icon when no records are found
     * 
     * @return string|null
     */
    protected function getEmptyStateIcon(): ?string
    {
        return 'heroicon-o-document-text';
    }

    /**
     * Get the empty state heading when no records are found
     * 
     * @return string|null
     */
    protected function getEmptyStateHeading(): ?string
    {
        return 'No inquiries found';
    }

    /**
     * Get the empty state description when no records are found
     * 
     * @return string|null
     */
    protected function getEmptyStateDescription(): ?string
    {
        return 'Create your first insurance inquiry to get started.';
    }

    /**
     * Get the empty state actions when no records are found
     * 
     * @return array
     */
    protected function getEmptyStateActions(): array
    {
        return [
            Actions\CreateAction::make()
                ->label('Create First Inquiry')
                ->icon('heroicon-o-plus')
                ->button(),
        ];
    }

    /**
     * Get the table records per page options
     * 
     * @return array
     */
    protected function getTableRecordsPerPageSelectOptions(): array
    {
        return [10, 25, 50, 100, 250];
    }

    /**
     * Get the default table records per page
     * 
     * @return int
     */
    protected function getDefaultTableRecordsPerPage(): int
    {
        return 25;
    }

    /**
     * Get the page heading
     * 
     * @return string
     */
    public function getHeading(): string
    {
        return 'Insurance Inquiries';
    }

    /**
     * Get the page subheading
     * 
     * @return string|null
     */
    public function getSubheading(): ?string
    {
        return 'Manage all insurance inquiries from this dashboard. Use filters to find specific records.';
    }

    /**
     * Get the breadcrumbs for the page
     * 
     * @return array
     */
    public function getBreadcrumbs(): array
    {
        return [
            route('filament.admin.pages.dashboard') => 'Dashboard',
            '' => 'Inquiries',
        ];
    }
}