<?php

namespace App\Enums;

enum TripPurpose: string
{
    case TOURISM = 'tourism';
    case BUSINESS = 'business';
    case EDUCATION = 'education';
    case MEDICAL = 'medical';
    case OTHER = 'other';

    public function label(): string
    {
        return match($this) {
            self::TOURISM => 'Tourism / Vacation',
            self::BUSINESS => 'Business',
            self::EDUCATION => 'Education / Study',
            self::MEDICAL => 'Medical Treatment',
            self::OTHER => 'Other',
        };
    }

    public function description(): string
    {
        return match($this) {
            self::TOURISM => 'Travel for leisure, vacation, or visiting friends/family',
            self::BUSINESS => 'Travel for work, meetings, conferences, or business activities',
            self::EDUCATION => 'Travel for studying, courses, training, or academic purposes',
            self::MEDICAL => 'Travel for medical treatment, surgery, or healthcare',
            self::OTHER => 'Other travel purposes not listed above',
        };
    }

    public function icon(): string
    {
        return match($this) {
            self::TOURISM => 'vacation',
            self::BUSINESS => 'business',
            self::EDUCATION => 'education',
            self::MEDICAL => 'medical',
            self::OTHER => 'other',
        };
    }

    public static function forSelect(): array
    {
        $cases = [];
        foreach (self::cases() as $case) {
            $cases[$case->value] = $case->label();
        }
        return $cases;
    }

    public static function withDescriptions(): array
    {
        $cases = [];
        foreach (self::cases() as $case) {
            $cases[] = [
                'value' => $case->value,
                'label' => $case->label(),
                'description' => $case->description(),
                'icon' => $case->icon(),
            ];
        }
        return $cases;
    }

    public static function getDefault(): self
    {
        return self::TOURISM;
    }

    public function isBusinessRelated(): bool
    {
        return $this === self::BUSINESS;
    }

    public function requiresAdditionalDocumentation(): bool
    {
        return match($this) {
            self::BUSINESS, self::EDUCATION, self::MEDICAL => true,
            default => false,
        };
    }

    public function getDocumentationRequirements(): array
    {
        return match($this) {
            self::BUSINESS => ['invitation_letter', 'business_card', 'conference_registration'],
            self::EDUCATION => ['acceptance_letter', 'student_id', 'course_schedule'],
            self::MEDICAL => ['medical_referral', 'hospital_appointment', 'treatment_plan'],
            default => [],
        };
    }
}