<?php

namespace App\Enums;

enum RateCategory: string
{
    case A = 'A';
    case B = 'B';
    case C = 'C';

    /**
     * Get the label for the rate category
     */
    public function label(): string
    {
        return match($this) {
            self::A => 'Category A (Lowest Risk)',
            self::B => 'Category B (Medium Risk)',
            self::C => 'Category C (Highest Risk)',
        };
    }

    /**
     * Get the description for the rate category
     */
    public function description(): string
    {
        return match($this) {
            self::A => 'All medical questions answered NO, or age under 60',
            self::B => 'Medium risk medical conditions',
            self::C => 'High risk medical conditions or automatic triggers',
        };
    }

    /**
     * Get the multiplier for premium calculation
     */
    public function premiumMultiplier(): float
    {
        return match($this) {
            self::A => 1.0,
            self::B => 1.5,
            self::C => 2.5,
        };
    }

    /**
     * Check if this rate category requires medical questionnaire
     */
    public function requiresQuestionnaire(): bool
    {
        return $this !== self::A;
    }

    /**
     * Get rate category based on age and medical conditions
     */
    public static function fromQuestionnaireAnswers(array $answers, int $age): self
    {
        // Age under 60 is always Category A
        if ($age < 60) {
            return self::A;
        }

        // Check for automatic Category C triggers
        $cTriggers = [
            'has_heart_condition' => true,
            'has_aortic_aneurysm' => true,
            'has_cirrhosis' => true,
            'has_parkinsons' => true,
            'has_alzheimers' => true,
            'high_blood_pressure_meds' => true,
        ];

        foreach ($cTriggers as $trigger => $value) {
            if (isset($answers[$trigger]) && $answers[$trigger] === 'yes') {
                return self::C;
            }
        }

        // Check for Category B triggers
        $bTriggers = [
            'gastrointestinal_bleeding' => true,
            'chronic_bowel_disorder' => true,
            'kidney_disorder' => true,
            'liver_disorder' => true,
            'pancreatitis' => true,
            'gallbladder_disorder' => true,
            'blood_disorder' => true,
            'recent_falls' => true,
            'frequent_er_visits' => true,
        ];

        foreach ($bTriggers as $trigger => $value) {
            if (isset($answers[$trigger]) && $answers[$trigger] === 'yes') {
                return self::B;
            }
        }

        // Default to Category A if all answers are NO
        return self::A;
    }

    /**
     * Get all cases for select dropdown
     */
    public static function forSelect(): array
    {
        $cases = [];
        foreach (self::cases() as $case) {
            $cases[$case->value] = $case->label();
        }
        return $cases;
    }

    /**
     * Get all cases with descriptions
     */
    public static function withDescriptions(): array
    {
        $cases = [];
        foreach (self::cases() as $case) {
            $cases[] = [
                'value' => $case->value,
                'label' => $case->label(),
                'description' => $case->description(),
                'premium_multiplier' => $case->premiumMultiplier(),
                'requires_questionnaire' => $case->requiresQuestionnaire(),
            ];
        }
        return $cases;
    }

    /**
     * Check if rate category is valid for given age
     */
    public function isValidForAge(int $age): bool
    {
        // Category A is valid for all ages
        if ($this === self::A) {
            return true;
        }

        // Categories B and C are only valid for age 60+
        return $age >= 60;
    }

    /**
     * Get the minimum age for this rate category
     */
    public function minimumAge(): ?int
    {
        return match($this) {
            self::A => null, // No minimum for Category A
            self::B => 60,
            self::C => 60,
        };
    }

    /**
     * Get rate category display color
     */
    public function color(): string
    {
        return match($this) {
            self::A => 'green',
            self::B => 'yellow',
            self::C => 'red',
        };
    }

    /**
     * Get rate category icon
     */
    public function icon(): string
    {
        return match($this) {
            self::A => 'check-circle',
            self::B => 'exclamation-triangle',
            self::C => 'exclamation-circle',
        };
    }
}